#include <string>
#include <iostream>
#include <fstream>
#include <cstring>
#include <algorithm>
#include <iomanip>
#include <regex>
#include "AminoAcidMapper.h"
#include "CmdLineParser.h"
#include "Util.h"
#include "Constants.h"

using namespace std;

void CmdLineParser::parseCmdLine(int nargs, char *args[], size_t &param_nsamples,
                                 string &param_mode, string &modelPath, string &type, char &samplingOption, vector<float>  &backbonePair) {

    if (nargs == 1) {
        coutHelp();
        exit(1);
    }

    if (strcmp(args[1], "--sample") == 0) {
        if (2 < nargs) {
            string optionName = args[2];
            if (optionName.size() == 1) {
                samplingOption = args[2][0];
            } else {
                isValidSamplingOption(optionName, samplingOption);
            }
        } else {
            cerr << "CmdLineError:: --sample input argument not provided" << endl;
            coutHelp();
            exit(1);
        }
    } else if (strcmp(args[1], "--help") == 0) {
        coutHelp();
        exit(0);
    } else if (strcmp(args[1], "--version") == 0) {
        coutVersion();
        exit(0);
    } else {
        coutHelp();
        exit(1);
    }

    switch (samplingOption) {
        case Constants::CONDITIONAL: {
            int mandatoryArgs = 0;
            for (int i = 3; i < nargs; i = i + 2) {
                if (args[i][0] == '-') {
                    if (strcmp(args[i], "--model") == 0) {
                        if (i + 1 < nargs) {
                            modelPath = args[i + 1];
                            isValidPath(modelPath, args[i]);
                            mandatoryArgs++;
                        }
                    } else if (strcmp(args[i], "--phi") == 0) {
                        if (i + 1 < nargs) {
                            float phi;
                            isValidAngle(args[i + 1], phi);
                            backbonePair[0] = Util::degreeToRadians(phi);
                            mandatoryArgs++;
                        }
                    } else if (strcmp(args[i], "--psi") == 0) {
                        if (i + 1 < nargs) {
                            float psi;
                            isValidAngle(args[i + 1], psi);
                            backbonePair[1] = Util::degreeToRadians(psi);
                            mandatoryArgs++;
                        }
                    }
                    else if (strcmp(args[i], "--nsamples") == 0) {
                        if (i + 1 < nargs) {
                            isValidSample(args[i + 1]);
                            param_nsamples = stoi(args[i + 1]);
                            mandatoryArgs++;
                        }
                    } else {
                        cerr << "CmdLineError:: Unsupported argument: " << args[i] << endl;
                        coutHelp();
                        exit(1);
                    }
                } else {
                    cerr << "CmdLineError:: Unsupported argument: " << args[i] << endl;
                    coutHelp();
                    exit(1);
                }
            }

            if (mandatoryArgs != 4) {
                cerr    << "CmdLineError:: Mandatory arguments not given. --model, --phi --psi, --nsamples are mandatory to run the application."
                        << endl;
                coutHelp();
                exit(1);
            }

            break;
        }
        case Constants::JOINT: {
            int mandatoryArgs = 0;
            for (int i = 3; i < nargs; i = i + 2) {
                if (args[i][0] == '-') {
                    if (strcmp(args[i], "--nsamples") == 0) {
                        if (i + 1 < nargs) {
                            isValidSample(args[i + 1]);
                            param_nsamples = stoi(args[i + 1]);
                            mandatoryArgs++;
                        }
                    } else if (strcmp(args[i], "--model") == 0) {
                        if (i + 1 < nargs) {
                            modelPath = args[i + 1];
                            isValidPath(modelPath, args[i]);
                            mandatoryArgs++;
                        }
                    } else {
                        cerr << "CmdLineError:: Unsupported argument: " << args[i] << endl;
                        coutHelp();
                        exit(1);
                    }
                } else {
                    cerr << "CmdLineError:: Unsupported argument: " << args[i] << endl;
                    coutHelp();
                    exit(1);
                }
            }

            if (mandatoryArgs != 2) {
                cerr
                        << "CmdLineError:: Mandatory arguments not given. --nsamples and --model are mandatory to run the application."
                        << endl;
                coutHelp();
                exit(1);
            }

            break;
        }
        default: {
            cerr << "CmdLineError:: --sample argument requires a valid sampling method. Following sampling options are currently supported. \n";
            cerr << "    joint(j) sample backbone and sidechain joint conformations" << endl;
            cerr << "    conditional(c) sample sidechain conformation conditioned on backbone conformation" << endl;
            exit(1);
        }
    }
}

void CmdLineParser::isValidSample(const string &nSamples) {
    bool isValidArg = !nSamples.empty() && find_if(nSamples.begin(),
                                                   nSamples.end(), [](unsigned char c) { return !std::isdigit(c); }) ==
                                           nSamples.end();
    if (!isValidArg) {
        cerr << "CmdLineError:: --nsamples argument requires a positive integer:\n";
        exit(1);
    }
}

void CmdLineParser::isValidMode(const string &mode) {

    if (Constants::MODE != mode) {
        cerr
                << "CmdLineError:: --mode argument requires a valid execution mode, fast option is currently supported. \n";
        cerr
                << "CmdLineError:: The application will fallback to default mode if argument is not provided. See README for further details"
                << endl;
        exit(1);
    }
}

void CmdLineParser::isValidType(const string &type) {

    if (Constants::RAD != type) {
        cerr << "CmdLineError:: --type argument requires a valid output type, rad option is currently supported. \n";
        cerr
                << "CmdLineError:: The application will fallback to default output type (degrees) if argument is not provided. See README for further details"
                << endl;
        exit(1);
    }
}

void CmdLineParser::isValidPath(const string &path, const string &arg) {
    ifstream infile(path);
    if (infile.fail()) {
        cout << "CmdLineError:: " << arg << " path does not point to a valid file. check your input path"
             << endl;
        exit(1);
    }
}

void CmdLineParser::isValidAngle(const char *data, float &backBoneAngle) {
    regex regex_num("^[+-]?(\\d*\\.)?\\d+$");

    if (regex_match(data, regex_num)) {
        backBoneAngle = atof(data);
        if (backBoneAngle <= 180 && backBoneAngle >= -180) {
            return;
        }
    }
    cerr << "CmdLineError:: --phi and --psi arguments require a angle [-180, 180) in degrees\n";
    exit(1);
}

void CmdLineParser::isValidSamplingOption(const string samplingOptionName, char &samplingOption) {
    if (samplingOptionName == "conditional") {
        samplingOption = Constants::CONDITIONAL;
    } else if (samplingOptionName == "joint") {
        samplingOption = Constants::JOINT;
    } else {
        samplingOption = samplingOptionName[0];
    }
}


void CmdLineParser::coutHelp() {
    cout << "\033[0;33;1;44m" << "   phisical-sampler" << "\033[0;0m";
    cout << "\033[0;32;1;44m (v"
         << Constants::VERSION_STRING
         << ")\033[0;0m";
    cout << "\033[0;37;1;44m"
         << " Sample from PhiSiCal (φψal) Library    "
         << "\033[0;0m";
    cout << "\033[0;33;1;44m" << "\033[0;0m\n";
    cout << "\033[0;1;37;1;45m";
    cout << "   reference: https://doi.org/10.1093/bioinformatics/btad25       "
         << "\033[0;0m";
    cout << left << setw(82);
    cout << endl;

    cout << "Usage:" << endl;
    cout << "To randomly sample JOINT <phi, psi, chi> dihedrals from PhiSiCal mixture model:" << endl;
    cout << "\tphisical-sampler --sample joint --model <model_path> --nsamples <n> " << endl;
    cout << endl;
    cout << "To randomly sample <chi> dihedrals from PhiSiCal mixture model, CONDITIONAL on specified <phi,psi>:" << endl;
    cout << "\tphisical-sampler --sample conditional --model <model_path> --nsamples <n> --phi <phi> --psi <psi>" << endl;
    cout << "(Refer README for details)." << endl;

}

void CmdLineParser::coutVersion() {
    cout << "\033[0;33;1;44m" << "   PhiSiCal" << "\033[0;0m";
    cout << "(v"
         << "\033[0;31;1;44m"
         << Constants::VERSION_STRING
         << "\033[0;37;1;44m"
         << ") " << "\033[0;0m";
    cout << "\033[0;37;1;44m";
    cout << setw(52)
         << " "
         << "\033[0;0m";
    cout << endl;
}
