#include <fstream>
#include <sstream>
#include <iostream>
#include <iomanip>
#include "FileHandler.h"
#include "AminoAcidMapper.h"
#include "Sampler.h"
#include "Util.h"
#include "Constants.h"

using namespace std;

vector<JointVonMisesCircular> FileHandler::readMixtureModel(const string &filePath, AminoAcid &aa, size_t &angleCount) {
    ifstream infile(filePath);
    string line;
    vector<JointVonMisesCircular> mixtureComponents;

    getline(infile, line); //read header
    aa = readAminoAcid(line);
    angleCount = AminoAcidMapper::getSideChainAngleCount(aa) + 2;

    getline(infile, line); //read header
    getline(infile, line); //read header
    getline(infile, line); //read header
    string angle;

    while (getline(infile, line)) {
        istringstream stream(line);

        vector<float> meanVector(angleCount);
        vector<float> kappaVector(angleCount);
        float weight;

        stream >> weight;
        for (int i = 0; i < angleCount; i++) {
            stream >> meanVector[i];
            stream >> kappaVector[i];
        }

        JointVonMisesCircular jointVonMisesCircular = {meanVector, kappaVector, weight};
        mixtureComponents.push_back(jointVonMisesCircular);
    }
    return mixtureComponents;
}

void FileHandler::outputDataFrame(const vector<vector<float>> &dataFrame, const size_t &angleCount,
                                  const string &type, const size_t &nsamples, const char &func) {

    if (!dataFrame.empty()) {
        string header = getHeader(angleCount, func);
        cout << header;

        if (type == Constants::DEG) {
            for (int i = 0; i < nsamples; ++i) {
                const auto &row = dataFrame[i];
                cout << setfill(' ') << right << fixed << setw(8) << setprecision(3) << Util::radiansToDegrees(row[0]);
                for (int j = 1; j < angleCount; j++) {
                    cout << setfill(' ') << right << fixed << setw(12) << setprecision(3)  <<Util::radiansToDegrees(row[j]);
                }
                cout << "\n";
            }
        } else {
            for (int i = 0; i < nsamples; ++i) {
                const auto &row = dataFrame[i];
                cout << setfill(' ') << right << fixed << setw(8) << setprecision(5) << row[0];
                for (int j = 1; j < angleCount; j++) {
                    cout << setfill(' ') << right << fixed << setw(12) << setprecision(5) << row[j];
                }
                cout << "\n";
            }
        }
    } else {
        cerr << "Not sufficient information to sample for the given input" << endl;
    }
}

string FileHandler::getHeader(const size_t &angles, const char &func) {
    stringstream str;

    if (func == Constants::JOINT) {
        size_t sideChainN = angles - 2;
        str <<  setfill(' ') << setw(8) << "phi" << right << setw(12) << "psi";
        for (int i = 0; i < sideChainN; ++i) {
            string sideChain = "chi" + to_string(i + 1);
            str << setw(12) << sideChain;
        }

    } else {
        str << setw(8) << "chi1";
        for (int i = 1; i < angles; ++i) {
            string sideChain = "chi" + to_string(i + 1);
            str << setw(12) << sideChain;
        }
    }

    string header = str.str() + "\n";
    return header;
}

AminoAcid FileHandler::readAminoAcid(const string &line) {
    string aa = line.substr(line.length() - 3);
    return AminoAcidMapper::getAminoAcidEnum(aa);
}